/******************************************************************************
 *
 * Copyright (c) 1999-2005 AppGate Network Security AB. All Rights Reserved.
 * 
 * This file contains Original Code and/or Modifications of Original Code as
 * defined in and that are subject to the MindTerm Public Source License,
 * Version 2.0, (the 'License'). You may not use this file except in compliance
 * with the License.
 * 
 * You should have received a copy of the MindTerm Public Source License
 * along with this software; see the file LICENSE.  If not, write to
 * AppGate Network Security AB, Otterhallegatan 2, SE-41118 Goteborg, SWEDEN
 *
 *****************************************************************************/

package com.mindbright.security.publickey;

import java.math.BigInteger;

import com.mindbright.jca.security.InvalidKeyException;
import com.mindbright.jca.security.interfaces.DSAParams;
import com.mindbright.jca.security.interfaces.DSAPublicKey;
import com.mindbright.jca.security.interfaces.DSAPrivateKey;

public class RawDSAWithSHA1 extends BaseSignature {

    public RawDSAWithSHA1() {
        super("SHA1");
    }

    protected void initVerify() throws InvalidKeyException {
        if(publicKey == null || !(publicKey instanceof DSAPublicKey)) {
            throw new InvalidKeyException("Wrong key for DSAWithSHA1 verify: " +
                                          publicKey);
        }
    }

    protected void initSign() throws InvalidKeyException {
        if(privateKey == null || !(privateKey instanceof DSAPrivateKey)) {
            throw new InvalidKeyException("Wrong key for DSAWithSHA1 sign: " +
                                          privateKey);
        }
    }

    protected byte[] sign(byte[] data) {
        DSAPrivateKey key  = (DSAPrivateKey)privateKey;
        DSAParams     parm = key.getParams();
        BigInteger    x    = key.getX();
        BigInteger    p    = parm.getP();
        BigInteger    q    = parm.getQ();
        BigInteger    g    = parm.getG();

        BigInteger[]  sign = DSAAlgorithm.sign(x, p, q, g, data);
        
        if (sign == null || sign.length != 2) {
            return null;
        }
        BigInteger r = sign[0];
        BigInteger s = sign[1];

        int dataSz = data.length;
        byte[] signature = new byte[dataSz * 2];
        byte[] tmp;

        tmp = unsignedBigIntToBytes(r, dataSz);
        System.arraycopy(tmp, 0, signature, 0, dataSz);

        tmp = unsignedBigIntToBytes(s, dataSz);
        System.arraycopy(tmp, 0, signature, dataSz, dataSz);

        return signature;
    }

    protected boolean verify(byte[] signature, byte[] data) {
        DSAPublicKey key  = (DSAPublicKey)publicKey;
        DSAParams    parm = key.getParams();
        BigInteger   y    = key.getY();
        BigInteger   p    = parm.getP();
        BigInteger   q    = parm.getQ();
        BigInteger   g    = parm.getG();

        int        dataSz = signature.length / 2;
        byte[]     ra     = new byte[dataSz];
        byte[]     sa     = new byte[dataSz];

        System.arraycopy(signature, 0, ra, 0, dataSz);
        System.arraycopy(signature, dataSz, sa, 0, dataSz);

        BigInteger r  = new BigInteger(1, ra);
        BigInteger s  = new BigInteger(1, sa);

        return DSAAlgorithm.verify(y, p, q, g, r, s, data);
    }

    private static byte[] unsignedBigIntToBytes(BigInteger bi, int size) {
        byte[] tmp  = bi.toByteArray();
        byte[] tmp2 = null;
        if(tmp.length > size) {
            tmp2 = new byte[size];
            System.arraycopy(tmp, tmp.length - size, tmp2, 0, size);
        } else if(tmp.length < size) {
            tmp2 = new byte[size];
            System.arraycopy(tmp, 0, tmp2, size - tmp.length, tmp.length);
        } else {
            tmp2 = tmp;
        }
        return tmp2;
    }

}
