/******************************************************************************
 *
 * Copyright (c) 1999-2005 AppGate Network Security AB. All Rights Reserved.
 * 
 * This file contains Original Code and/or Modifications of Original Code as
 * defined in and that are subject to the MindTerm Public Source License,
 * Version 2.0, (the 'License'). You may not use this file except in compliance
 * with the License.
 * 
 * You should have received a copy of the MindTerm Public Source License
 * along with this software; see the file LICENSE.  If not, write to
 * AppGate Network Security AB, Otterhallegatan 2, SE-41118 Goteborg, SWEDEN
 *
 *****************************************************************************/

package com.mindbright.jce.crypto;

import com.mindbright.jca.security.InvalidKeyException;
import com.mindbright.jca.security.Key;
import com.mindbright.jca.security.Provider;
import com.mindbright.jca.security.ProviderLookup;
import com.mindbright.jca.security.spec.AlgorithmParameterSpec;
// import com.mindbright.jca.security.AlgorithmParameters;
import com.mindbright.jca.security.NoSuchAlgorithmException;
import com.mindbright.jca.security.NoSuchProviderException;

public class Cipher {

    public final static int DECRYPT_MODE = 1;
    public final static int ENCRYPT_MODE = 2;

    CipherSpi cipherSpi;
    Provider  provider;
    String    transformation;
    String    algorithm;

    protected Cipher(CipherSpi cipherSpi, Provider provider,
                     String transformation) {
        this.cipherSpi      = cipherSpi;
        this.provider       = provider;
        this.transformation = transformation;
        int i = transformation.indexOf('/');
        if(i != -1) {
            this.algorithm = transformation.substring(0, i);
        } else {
            this.algorithm = transformation;
        }
    }

    /**
     * Encrypt the entire input array.
     * <p>
     * Note that the input must be a multiple of the block size bytes long.
     *
     * @param input array of data to encrypt
     * @return the encrypted data
     */
    public final byte[] doFinal(byte[] input) {
        return doFinal(input, 0, input.length);
    }

    /**
     * Encrypt part of the input array
     * <p>
     * Note that the input must be a multiple of the block size bytes long.
     *
     * @param input       array of data to encrypt
     * @param inputOffset start of data to encrypt
     * @param inputLen    length of data to encrypt
     * @return the encrypted data
     */
    public final byte[] doFinal(byte[] input, int inputOffset, int inputLen) {
        byte[] output = new byte[getOutputSize(inputLen)];
        doFinal(input, inputOffset, inputLen, output, 0);
        return output;
    }

    /**
     * Encrypt part of the input array and stores the result in the
     * given output array. 
     * <p>
     * Note that the input must be a multiple of the block size bytes long.
     *
     * @param input       array of data to encrypt
     * @param inputOffset start of data to encrypt
     * @param inputLen    length of data to encrypt
     * @param output      array in which output is stored. This array
     *                    must have room for at least
     *                    getOutputSize(inputLen) bytes
     * @return the number of bytes store in the output array
     */
    public final int doFinal(byte[] input, int inputOffset, int inputLen,
                             byte[] output) {
        return doFinal(input, inputOffset, inputLen, output, 0);
    }

    /**
     * Encrypt part of the input array and stores the result in the
     * given output array. 
     * <p>
     * Note that the input must be a multiple of the block size bytes long.
     *
     * @param input        array of data to encrypt
     * @param inputOffset  start of data to encrypt
     * @param inputLen     length of data to encrypt
     * @param output       array in which output is stored. This array
     *                     must have room for at least
     *                     getOutputSize(inputLen) bytes
     * @param outputOffset Start offset in the output array where the output
     *                     should be stored.
     * @return the number of bytes store in the output array
     */
    public final int doFinal(byte[] input, int inputOffset, int inputLen,
                             byte[] output, int outputOffset) {
        return cipherSpi.engineDoFinal(input, inputOffset, inputLen,
                                       output, outputOffset);
    }

    /**
     * Get the name of the algorithm implemented by this Cipher
     * instance.
     *
     * @return algorithm name
     */
    public final String getAlgorithm() {
        return algorithm;
    }

    /**
     * Get how big blocks this algorithm works on.
     *
     * @return the block size (in bytes)
     */
    public final int getBlockSize() {
        return cipherSpi.engineGetBlockSize();
    }

    /**
     * Get a Cipher instance which implements the given algorithm in
     * any of the registered providers.
     *
     * @param transformation name of the desired algorithm (like
     * AES/CBC).
     * @return a Cipher instance implementing the algorithm
     * @throws NoSuchAlgorithmException if no implementation can be found
     */
    public static final Cipher getInstance(String transformation)
    throws NoSuchAlgorithmException {
        try {
            String provider =
                ProviderLookup.findImplementingProvider("Cipher",
                                                        transformation);
            return getInstance(transformation, provider);
        } catch (NoSuchProviderException e) {
            throw new Error("Error in Cipher: " + e);
        }
    }

    /**
     * Get a Cipher instance which implements the given algorithm in
     * the given provider.
     *
     * @param transformation name of the desired algorithm (like
     * AES/CBC).
     * @return a Cipher instance implementing the algorithm
     * @throws NoSuchAlgorithmException if no implementation can be found
     */
    public static final Cipher getInstance(String transformation,
                                           String provider)
    throws NoSuchAlgorithmException, NoSuchProviderException {
        ProviderLookup pl = ProviderLookup.getImplementation("Cipher",
                            transformation,
                            provider);

        CipherSpi cipherSpi = (CipherSpi)pl.getImpl();

        int i = transformation.indexOf('/');
        if(i != -1) {
            int j = transformation.indexOf('/', i + 1);
            if(j != -1) {
                cipherSpi.engineSetPadding(transformation.substring(j + 1));
            } else {
                j = transformation.length();
            }
            cipherSpi.engineSetMode(transformation.substring(i + 1, j));
        }

        return new Cipher(cipherSpi, pl.getProvider(), transformation);
    }

    /**
     * Get the IV used by this instance
     */
    public final byte[] getIV() {
        return cipherSpi.engineGetIV();
    }

    /**
     * Calculate how much encryted data an imput of the given length
     * will cause.
     *
     * @param inputLen length of input data
     * @return number of bytes output
     */
    public final int getOutputSize(int inputLen) {
        return cipherSpi.engineGetOutputSize(inputLen);
    }

    /**
     * Get the provider which implements this algorithm.
     */
    public final Provider getProvider() {
        return provider;
    }

    /**
     * Initialize this Cipher instance.
     *
     * @param opmode The operating mode. This should be either
     * Cipher.ENCRYPT_MODE or Cipher.DECRYPT_MODE.
     * @param key the encryption or decryption key
     */
    public final void init(int opmode, Key key) throws InvalidKeyException {
        init(opmode, key, null);
    }

    /**
     * Initialize this Cipher instance.
     *
     * @param opmode The operating mode. This should be either
     * Cipher.ENCRYPT_MODE or Cipher.DECRYPT_MODE.
     * @param key the encryption or decryption key
     * @param params Extra algorithm parameters
     */
    public final void init(int opmode, Key key, AlgorithmParameterSpec params)
    throws InvalidKeyException {
        cipherSpi.engineInit(opmode, key, params, null);
    }
}
