/**
 * com.mckoi.database.WorkerPool  12 Aug 2000
 *
 * Mckoi SQL Database ( http://www.mckoi.com/database )
 * Copyright (C) 2000, 2001, 2002  Diehl and Associates, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * Version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License Version 2 for more details.
 *
 * You should have received a copy of the GNU General Public License
 * Version 2 along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Change Log:
 * 
 * 
 */

package com.mckoi.database;

import com.mckoi.debug.DebugLogger;
import java.util.LinkedList;

/**
 * Maintains a pool of worker threads that are used to dispatch commands to
 * a Database sub-system.
 *
 * @author Tobias Downer
 */

final class WorkerPool {

  /**
   * The TransactionSystem that this pool is part of.
   */
  private TransactionSystem system;

  /**
   * This is the maximum number of worker threads that will be created.
   */
  private int MAXIMUM_WORKER_THREADS = 4;

  /**
   * This is a queue of 'WorkerThread' objects that are currently available
   * to process commands from the service providers.
   */
  private LinkedList available_worker_threads;

  /**
   * The number of worker threads that have been created in total.
   */
  private int worker_thread_count;

  /**
   * A list of pending Runnable objects that are due to be executed.  This is
   * a queue of events to be run.
   */
  private LinkedList run_queue;

  /**
   * If this is set to false, then no commands will be executed by the
   * 'execute' method.
   */
  private boolean is_executing_commands;


  /**
   * Constructs the worker thread pool.
   */
  WorkerPool(TransactionSystem system, int max_worker_threads) {
    this.system = system;
    MAXIMUM_WORKER_THREADS = max_worker_threads;

    is_executing_commands = false;

    // Set up the run queue
    run_queue = new LinkedList();
    // Set up the worker threads
    available_worker_threads = new LinkedList();
    worker_thread_count = 0;
//    // Create a single worker thread and start it.
//    ++worker_thread_count;
//    WorkerThread wt = new WorkerThread(this);
//    wt.start();

  }

  /**
   * Returns a DebugLogger object that we can use to log debug messages.
   */
  public final DebugLogger Debug() {
    return system.Debug();
  }

  // ---------- Thread Pooling methods ----------

  /**
   * This is called by a WorkerThread when it is decided that it is ready
   * to service a new command.
   */
  void notifyWorkerReady(WorkerThread worker_thread) {
    synchronized (available_worker_threads) {
      // Add it to the queue of worker threads that are available.
      available_worker_threads.add(worker_thread);

      // Are there any commands pending?
      int q_len = run_queue.size();
      if (q_len > 0) {
        // Execute the bottom element on the queue
        RunCommand command = (RunCommand) run_queue.remove(0);
        execute(command.user, command.database, command.runnable);
      }
    }
  }

  /**
   * This returns the first available WorkerThread object from the thread
   * pool.  If there are no available worker threads available then it returns
   * null.  This method must execute fast and must not block.
   */
  private WorkerThread getFirstWaitingThread() {
    synchronized (available_worker_threads) {
      // Is there a worker thread available?
      int size = available_worker_threads.size();
      if (size > 0) {
        // Yes so remove the first element and return it.
        WorkerThread wt = (WorkerThread) available_worker_threads.remove(0);
        return wt;
      }
      else {
        // Otherwise create a new worker thread if we can.
        if (worker_thread_count < MAXIMUM_WORKER_THREADS) {
          ++worker_thread_count;
          WorkerThread wt = new WorkerThread(this);
          wt.start();
          // NOTE: We must _not_ return the worker thread we have just created.
          //   We must wait until the worker thread has made it self known by
          //   it calling the 'notifyWorkerReady' method.
        }
        return null;
      }
    }
  }

  /**
   * Executes database functions from the 'run' method of the given runnable
   * instance on a worker thread.  All database functions should go through
   * a worker thread.  If we ensure this, we can easily stop all database
   * functions from executing.  Also, we only need to have a certain number
   * of threads active at any one time rather than a unique thread for each
   * connection.
   */
  void execute(User user, DatabaseConnection database, Runnable runner) {
    synchronized (available_worker_threads) {
      if (is_executing_commands) {
        WorkerThread worker = getFirstWaitingThread();
        if (worker != null) {
//          System.out.println("[Database] executing runner");
          worker.execute(user, database, runner);
          return;
        }
      }
//      System.out.println("[Database] adding to run queue");
      RunCommand command = new RunCommand(user, database, runner);
      run_queue.add(command);
    }
  }

  /**
   * Controls whether the database is allowed to execute commands or not.  If
   * this is set to true, then calls to 'execute' will make commands execute.
   */
  void setIsExecutingCommands(boolean status) {
    synchronized (available_worker_threads) {
      if (status == true) {
        is_executing_commands = true;

        // Execute everything on the queue
        for (int i = run_queue.size() - 1; i >= 0; --i) {
          RunCommand command = (RunCommand) run_queue.remove(i);
          execute(command.user, command.database, command.runnable);
        }
      }
      else {
        is_executing_commands = false;
      }
    }
  }

  /**
   * Waits until all executing commands have stopped.  This is best called
   * right after a call to 'setIsExecutingCommands(false)'.  If these two
   * commands are run, the database is in a known state where no commands
   * can be executed.
   * <p>
   * NOTE: This can't be called from the WorkerThread.  Deadlock will
   *   result if we were allowed to do this.
   */
  void waitUntilAllWorkersQuiet() {
    if (Thread.currentThread() instanceof WorkerThread) {
      throw new Error("Can't call this method from a WorkerThread!");
    }

    synchronized (available_worker_threads) {
      // loop until available works = total worker thread count.
      while (worker_thread_count != available_worker_threads.size()) {
        // Wait half a second
        try {
          available_worker_threads.wait(500);
        }
        catch (InterruptedException e) {}
        // ISSUE: If this lasts for more than 10 minutes, one of the worker
        //   threads is likely in a state of deadlock.  If this happens, we
        //   should probably find all the open worker threads and clean them
        //   up nicely.
      }
    }
  }

  /**
   * Shuts down the WorkerPool object stopping all worker threads.
   */
  void shutdown() {
    synchronized (available_worker_threads) {
      while (available_worker_threads.size() > 0) {
        WorkerThread wt = (WorkerThread) available_worker_threads.remove(0);
        --worker_thread_count;
        wt.shutdown();
      }
    }
  }

  // ---------- Inner classes ----------

  /**
   * Structures within the run_queue list.  This stores the Runnable to
   * run and the User that's executing the command.
   */
  private static final class RunCommand {
    User     user;
    DatabaseConnection database;
    Runnable runnable;
    public RunCommand(User user, DatabaseConnection database,
                      Runnable runnable) {
      this.user = user;
      this.database = database;
      this.runnable = runnable;
    }
  }

}
