/**
 * com.mckoi.database.ReferenceTable  03 Apr 1998
 *
 * Mckoi SQL Database ( http://www.mckoi.com/database )
 * Copyright (C) 2000, 2001, 2002  Diehl and Associates, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * Version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License Version 2 for more details.
 *
 * You should have received a copy of the GNU General Public License
 * Version 2 along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Change Log:
 * 
 * 
 */

package com.mckoi.database;

import com.mckoi.util.IntegerVector;

/**
 * This is an implementation of a Table that references a DataTable as its
 * parent.  This is a one-to-one relationship unlike the VirtualTable class
 * which is a one-to-many relationship.
 * <p>
 * The entire purpose of this class is as a filter.  We can use it to rename
 * a DataTable class to any domain we feel like.  This allows us to generate
 * unique column names.
 * <p>
 * For example, say we need to join the same table.  We can use this method
 * to ensure that the newly joined table won't have duplicate column names.
 * <p>
 * This object implements RootTable.
 *
 * @author Tobias Downer
 */

public final class ReferenceTable extends FilterTable implements RootTable {

  /**
   * This represents the new name of the table.
   */
  private TableName table_name;

  /**
   * The modified DataTableDef object for this reference.
   */
  private DataTableDef modified_table_def;

  
  /**
   * The Constructor.
   */
  ReferenceTable(Table table, TableName tname) {
    super(table);
    table_name = tname;

    // Create a modified table def based on the parent def.
    modified_table_def = new DataTableDef(table.getDataTableDef());
    modified_table_def.setTableName(tname);
    modified_table_def.setImmutable();
  }

  /**
   * Constructs the ReferenceTable given the parent table, and a new
   * DataTableDef that describes the columns in this table.  This is used if
   * we want to redefine the column names.
   * <p>
   * Note that the given DataTableDef must contain the same number of columns as
   * the parent table, and the columns must be the same type.
   */
  ReferenceTable(Table table, DataTableDef def) {
    super(table);
    table_name = def.getTableName();
    
    modified_table_def = def;
  }
  
  /**
   * Filters the name of the table.  This returns the declared name of the
   * table.
   */
  public TableName getTableName() {
    return table_name;
  }

  /**
   * Returns the 'modified' DataTableDef object for this reference.
   */
  public DataTableDef getDataTableDef() {
    return modified_table_def;
  }

  /**
   * Given a fully qualified variable field name, ie. 'APP.CUSTOMER.CUSTOMERID'
   * this will return the column number the field is at.  Returns -1 if the
   * field does not exist in the table.
   */
  public int findFieldName(Variable v) {
    TableName table_name = v.getTableName();
    if (table_name != null && table_name.equals(getTableName())) {
      return getDataTableDef().fastFindColumnName(v.getName());
    }
    return -1;
  }

  /**
   * Returns a fully qualified Variable object that represents the name of
   * the column at the given index.  For example,
   *   new Variable(new TableName("APP", "CUSTOMER"), "ID")
   */
  public Variable getResolvedVariable(int column) {
    return new Variable(getTableName(),
                        getDataTableDef().columnAt(column).getName());
  }

  public boolean typeEquals(RootTable table) {
    return (this == table);
  }

}
