/**
 * com.mckoi.database.DatabaseConnection  21 Nov 2000
 *
 * Mckoi SQL Database ( http://www.mckoi.com/database )
 * Copyright (C) 2000, 2001, 2002  Diehl and Associates, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * Version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License Version 2 for more details.
 *
 * You should have received a copy of the GNU General Public License
 * Version 2 along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Change Log:
 * 
 * 
 */

package com.mckoi.database;

import com.mckoi.debug.*;
import com.mckoi.util.Cache;
import com.mckoi.database.global.ByteLongObject;
import com.mckoi.database.global.Ref;
import com.mckoi.database.jdbc.SQLQuery;
import java.util.HashMap;
import java.util.ArrayList;
import java.math.BigDecimal;

/**
 * An object that represents a connection to a Database.  This object handles
 * all transactional queries and modifications to the database.
 *
 * @author Tobias Downer
 */

public class DatabaseConnection implements TriggerListener {

  /**
   * The User that this connection has been made by.
   */
  private User user;

  /**
   * The Database object that this connection is on.
   */
  private Database database;

  /**
   * The DebugLogger object that we can use to log messages to.
   */
  private DebugLogger logger;

  /**
   * A loop-back object that is managing this connection.  This typically is
   * the session protocol.  This is notified of all connection events, such as
   * triggers.
   */
  private CallBack call_back;

  /**
   * The locking mechanism within this connection.
   */
  private LockingMechanism locking_mechanism;

  /**
   * The TableDataConglomerate object that is used for transactional access
   * to the data.
   */
  private TableDataConglomerate conglomerate;

  /**
   * The current Transaction that this connection is operating within.
   */
  private Transaction transaction;

  /**
   * The current java.sql.Connection object that can be used to access the
   * transaction internally.
   */
  private java.sql.Connection jdbc_connection;

  /**
   * A HashMap of DataTable objects that have been created within this
   * connection.
   */
  private HashMap tables_cache;

  /**
   * A buffer of triggers.  This contains triggers that can't fire until
   * the current transaction has closed.  These triggers were generated by
   * external actions outside of the context of this transaction.
   */
  private ArrayList trigger_event_buffer;

  /**
   * A list of triggers that are fired by actions taken on tables in this
   * transaction.  When the transaction is successfully committed, these
   * trigger events need to be propogated to other connections in the database
   * listening for trigger events on the triggered objects.
   */
  private ArrayList trigger_event_list;
  
  /**
   * If this is true then the database connection is in 'auto-commit' mode.
   * This implies a COMMIT instruction is executed after every complete
   * statement in the language grammar.  By default this is true.
   */
  private boolean auto_commit;

  /**
   * The current transaction isolation level this connect is operating under.
   * 1 = READ UNCOMMITTED, 2 = READ COMMITTED, 3 = REPEATABLE READ,
   * 4 = SERIALIZABLE.
   */
  private int transaction_isolation;

  /**
   * A flag which if set to true, will not allow 'commit' or 'rollback'
   * commands on the transaction to occur and therefore prevent any open
   * transaction from closing.  This is useful for restricting the ability
   * of a stored procedure to close.
   */
  private boolean close_transaction_disabled;
  
  /**
   * The name of the schema that this connection is currently in.  If the
   * schema is "" then this connection is in the default schema (effectively
   * no schema).
   */
  private String current_schema;

  /**
   * The GrantManager object for this connection.
   */
  private GrantManager grant_manager;

  /**
   * The procedure manager object for this connection.
   */
  private ProcedureManager procedure_manager;

  /**
   * The connection trigger manager that handles actions that cause triggers
   * to fire on this connection.
   */
  private ConnectionTriggerManager connection_trigger_manager;

  /**
   * The connection view manager that handles view information through this
   * connection.
   */
  private ViewManager view_manager;
  
  /**
   * The list of all TableBackedCache objects that have been attached to this
   * DatabaseConnection and are to be notified of transaction start/end
   * events.
   */
  private ArrayList table_backed_cache_list;

  /**
   * A local member that represents the static list of internal tables
   * that represent connection specific properties such as username,
   * connection, statistics, etc.
   */
  private ConnectionInternalTableInfo connection_internal_table_info;
  
  // ----- Local flags -----

  /**
   * True if transactions through this connection generate an error when
   * there is a dirty select on a table.
   */
  private boolean error_on_dirty_select;

  /**
   * True if this connection resolves identifiers case insensitive.
   */
  private boolean case_insensitive_identifiers;

  // ----- OLD and NEW table information for triggers -----

  /**
   * A local member that represents the OLD and NEW system tables that
   * represent the OLD and NEW data in a triggered action.
   */
  private OldAndNewInternalTableInfo old_new_table_info;

  /**
   * The current state of the OLD and NEW system tables including any cached
   * information about the tables.
   */
  private OldNewTableState current_old_new_state = new OldNewTableState();



  /**
   * (package protected) Constructs the connection.
   */
  DatabaseConnection(Database database, User user, CallBack call_back) {
    this.database = database;
    this.user = user;
    this.logger = database.Debug();
    this.call_back = call_back;
    this.conglomerate = database.getConglomerate();
    this.locking_mechanism = new LockingMechanism(Debug());
    this.trigger_event_buffer = new ArrayList();
    this.trigger_event_list = new ArrayList();
    tables_cache = new HashMap();
    auto_commit = true;

    current_schema = Database.DEFAULT_SCHEMA;
    this.close_transaction_disabled = false;

    this.table_backed_cache_list = new ArrayList();

    connection_internal_table_info = new ConnectionInternalTableInfo();
    old_new_table_info = new OldAndNewInternalTableInfo();

    error_on_dirty_select =
                          database.getSystem().transactionErrorOnDirtySelect();
    case_insensitive_identifiers = database.getSystem().ignoreIdentifierCase();

  }

  /**
   * Initializes this DatabaseConnection (possibly by initializing state from
   * the database).
   */
  void init() {
    // Create the grant manager for this connection.
    grant_manager = new GrantManager(this);
    // Create the procedure manager for this connection.
    procedure_manager = new ProcedureManager(this);
    // Create the connection trigger manager object
    connection_trigger_manager = new ConnectionTriggerManager(this);
    // Create the view manager
    view_manager = new ViewManager(this);
  }
  
  /**
   * Returns the transaction.  If 'transaction' is null then it opens a
   * new transaction within the conglomerate.
   */
  private Transaction getTransaction() {
    synchronized (this) {
      if (transaction == null) {
        transaction = conglomerate.createTransaction();
        transaction.setErrorOnDirtySelect(error_on_dirty_select);
        // Internal tables (connection statistics, etc)
        transaction.addInternalTableInfo(connection_internal_table_info);
        // OLD and NEW system tables (if applicable)
        transaction.addInternalTableInfo(old_new_table_info);
        // Model views as tables (obviously)
        transaction.addInternalTableInfo(
              ViewManager.createInternalTableInfo(view_manager, transaction));
        // Model procedures as tables
        transaction.addInternalTableInfo(
              ProcedureManager.createInternalTableInfo(transaction));
        // Model sequences as tables
        transaction.addInternalTableInfo(
              SequenceManager.createInternalTableInfo(transaction));
        // Model triggers as tables
        transaction.addInternalTableInfo(
              ConnectionTriggerManager.createInternalTableInfo(transaction));

        // Notify any table backed caches that this transaction has started.
        int sz = table_backed_cache_list.size();
        for (int i = 0; i < sz; ++i) {
          TableBackedCache cache =
                           (TableBackedCache) table_backed_cache_list.get(i);
          cache.transactionStarted();
        }

      }
    }
    return transaction;
  }

  /**
   * Returns a freshly deserialized QueryPlanNode object for the given view
   * object.
   */
  QueryPlanNode createViewQueryPlanNode(TableName table_name) {
    return view_manager.createViewQueryPlanNode(table_name);
  }
  
  /**
   * Returns a java.sql.Connection object that can be used as a JDBC
   * interface to access the current transaction of this DatabaseConnection.
   * <p>
   * There are a few important considerations when using the JDBC connection;
   * <ul>
   *   <li>The returned Connection does not allow auto-commit to be set.  It
   *       is intended to be used to issue commands to this
   *       DatabaseConnection from inside a transaction so auto-commit does
   *       not make sense.
   *   <li>The returned object must only be accessed from the same worker
   *       thread that is currently accessing this DatabaseConnection.  The
   *       returned Connection is <b>NOT</b> multi-thread capable.
   *   <li>The java.sql.Connection returned here is invalidated (disposed) when
   *       the current transaction is closed (committed or rolled back).
   *   <li>This method returns the same java.sql.Connection on multiple calls
   *       to this method (while a transaction is open).
   *   <li>The DatabaseConnection must be locked in EXCLUSIVE mode or the
   *       queries will fail.
   * </ul>
   */
  public java.sql.Connection getJDBCConnection() {
    if (jdbc_connection == null) {
      jdbc_connection =
                     InternalJDBCHelper.createJDBCConnection(getUser(), this);
    }
    return jdbc_connection;
  }

  /**
   * Creates an object that implements ProcedureConnection that provides access
   * to this connection.
   * <p>
   * Note that this connection is set to the user of the privs that the
   * procedure executes under when this method returns.
   * <p>
   * There must be a 100% guarentee that after this method is called, a call to
   * 'disposeProcedureConnection' is called which cleans up the state of this
   * object.
   */
  ProcedureConnection createProcedureConnection(User user) {
    // Create the ProcedureConnection object,
    DCProcedureConnection c = new DCProcedureConnection();
    // Record the current user
    c.previous_user = getUser();
    // Record the current 'close_transaction_disabled' flag
    c.transaction_disabled_flag = close_transaction_disabled;
    // Set the new user
    setUser(user);
    // Disable the ability to close a transaction
    close_transaction_disabled = true;
    // Return
    return c;
  }

  /**
   * Disposes of the ProcedureConnection that was previously created by the
   * 'createProcedure' method.
   */
  void disposeProcedureConnection(ProcedureConnection connection) {
    DCProcedureConnection c = (DCProcedureConnection) connection;
    // Revert back to the previous user.
    setUser(c.previous_user);
    // Revert back to the previous transaction disable status.
    close_transaction_disabled = c.transaction_disabled_flag;
    // Dispose of the connection
    c.dispose();
  }

  /**
   * Returns the DatabaseSystem object for this connection.
   */
  public DatabaseSystem getSystem() {
    return database.getSystem();
  }

  /**
   * Returns the Database object for this connection.
   */
  public Database getDatabase() {
    return database;
  }

  /**
   * Returns the conglomerate of this connection.
   */
  TableDataConglomerate getConglomerate() {
    return conglomerate;
  }
  
  /**
   * Sets the User object for this connection.  This is necessary because we
   * may want to temporarily change the user on this connection to allow
   * top level queries in a different privilege space.
   */
  void setUser(User user) {
    this.user = user;
  }
  
  /**
   * Returns the User object for this connection.
   */
  public User getUser() {
    return user;
  }

  /**
   * Returns a DebugLogger object that we can use to log debug messages to.
   */
  public final DebugLogger Debug() {
    return logger;
  }

  /**
   * Returns the connection trigger manager for this connection.
   */
  public ConnectionTriggerManager getConnectionTriggerManager() {
    return connection_trigger_manager;
  }
  
  /**
   * Returns the GrantManager object that manages grants for tables in the
   * database for this connection/user.
   */
  public GrantManager getGrantManager() {
    return grant_manager;
  }

  /**
   * Returns the ProcedureManager object that manages database functions and
   * procedures in the database for this connection/user.
   */
  public ProcedureManager getProcedureManager() {
    return procedure_manager;
  }
  
  /**
   * Sets the auto-commit mode.
   */
  public void setAutoCommit(boolean status) {
    auto_commit = status;
  }

  /**
   * Sets the transaction isolation level from a string.
   */
  public void setTransactionIsolation(String name) {
    if (name.equals("serializable")) {
      transaction_isolation = 4;
    }
    else {
      throw new Error("Can not set transaction isolation to " + name);
    }
  }

  /**
   * Assigns a variable to the expression for this connection.  This is a
   * generic way of setting properties of the connection.  Currently supported
   * variables are;
   * <p>
   * ERROR_ON_DIRTY_SELECT - set to Boolean.TRUE for turning this transaction
   *   conflict off on this connection.
   * CASE_INSENSITIVE_IDENTIFIERS - Boolean.TRUE means the grammar becomes
   *   case insensitive for identifiers resolved by the grammar.
   */
  public void setVar(String name, Expression exp) {
    if (name.toUpperCase().equals("ERROR_ON_DIRTY_SELECT")) {
      error_on_dirty_select = toBooleanValue(exp);
    }
    else if (name.toUpperCase().equals("CASE_INSENSITIVE_IDENTIFIERS")) {
      case_insensitive_identifiers = toBooleanValue(exp);
    }
  }

  /**
   * Evaluates the expression to a boolean value (true or false).
   */
  private static boolean toBooleanValue(Expression exp) {
    Boolean b = exp.evaluate(null, null, null).toBoolean();
    if (b == null) {
      throw new StatementException(
                 "Expression does not evaluate to a boolean (true or false).");
    }
    return b.booleanValue();
  }

  /**
   * Returns the auto-commit status of this connection.  If this is true then
   * the language layer must execute a COMMIT after every statement.
   */
  public boolean getAutoCommit() {
    return auto_commit;
  }

  /**
   * Returns the transaction isolation level of this connection.
   */
  public int getTransactionIsolation() {
    return transaction_isolation;
  }

  /**
   * Returns the transaction isolation level of this connection as a string.
   */
  public String getTransactionIsolationAsString() {
    int il = getTransactionIsolation();
    if (il == 1) {
      return "read uncommitted";
    }
    else if (il == 2) {
      return "read committed";
    }
    else if (il == 3) {
      return "repeatable read";
    }
    else if (il == 4) {
      return "serializable";
    }
    else {
      return "unknown isolation level";
    }
  }

  /**
   * Returns the name of the schema that this connection is within.
   */
  public String getCurrentSchema() {
    return current_schema;
  }

  /**
   * Returns true if the connection is in case insensitive mode.  In case
   * insensitive mode the case of identifier strings is not important.
   */
  public boolean isInCaseInsensitiveMode() {
    return case_insensitive_identifiers;
  }

  /**
   * Sets the schema that this connection is within.
   */
  public void setCurrentSchema(String current_schema) {
    this.current_schema = current_schema;
  }

  /**
   * Returns the LockingMechanism object that is within the context of this
   * database connection.  This manages read/write locking within this
   * connection.
   */
  public LockingMechanism getLockingMechanism() {
    return locking_mechanism;
  }

  /**
   * Attaches a TableBackedCache object to this DatabaseConnection which is
   * notified when a transaction is started and stopped, and when the table
   * being backed has changes made to it.
   */
  void attachTableBackedCache(TableBackedCache cache) {
    cache.attachTo(conglomerate);
    table_backed_cache_list.add(cache);
  }
  
  /**
   * Returns a TableName[] array that contains the list of database
   * tables that are visible by this transaction.
   * <p>
   * This returns the list of all objects that represent queriable tables in
   * the database.
   */
  public TableName[] getTableList() {
    return getTransaction().getTableList();
  }

  /**
   * Returns true if the table exists within this connection transaction.
   */
  public boolean tableExists(String table_name) {
    return tableExists(new TableName(current_schema, table_name));
  }

  /**
   * Returns true if the table exists within this connection transaction.
   */
  public boolean tableExists(TableName table_name) {
    table_name = substituteReservedTableName(table_name);
    return getTransaction().tableExists(table_name);
  }

  /**
   * Returns the type of the table object.  Currently this is either "TABLE"
   * or "VIEW".
   */
  public String getTableType(TableName table_name) {
    table_name = substituteReservedTableName(table_name);
    return getTransaction().getTableType(table_name);
  }

  /**
   * Attempts to resolve the given table name to its correct case assuming
   * the table name represents a case insensitive version of the name.  For
   * example, "aPP.CuSTOMer" may resolve to "APP.Customer".  If the table
   * name can not resolve to a valid identifier it returns the input table
   * name, therefore the actual presence of the table should always be
   * checked by calling 'tableExists' after this method returns.
   */
  public TableName tryResolveCase(TableName table_name) {
    table_name = substituteReservedTableName(table_name);
    table_name = getTransaction().tryResolveCase(table_name);
    return table_name;
  }

  /**
   * Resolves a TableName string (eg. 'Customer' 'APP.Customer' ) to a
   * TableName object.  If the schema part of the table name is not present
   * then it is set to the current schema of the database connection.  If the
   * database is ignoring the case then this will correctly resolve the table
   * to the cased version of the table name.
   */
  public TableName resolveTableName(String name) {
    TableName table_name = TableName.resolve(getCurrentSchema(), name);
    table_name = substituteReservedTableName(table_name);
    if (isInCaseInsensitiveMode()) {
      // Try and resolve the case of the table name,
      table_name = tryResolveCase(table_name);
    }
    return table_name;
  }

  /**
   * Resolves the given string to a table name, throwing an exception if
   * the reference is ambiguous.  This also generates an exception if the
   * table object is not found.
   */
  public TableName resolveToTableName(String name) {
    TableName table_name = TableName.resolve(getCurrentSchema(), name);
    if (table_name.getName().equalsIgnoreCase("OLD")) {
      return Database.OLD_TRIGGER_TABLE;
    }
    else if (table_name.getName().equalsIgnoreCase("NEW")) {
      return Database.NEW_TRIGGER_TABLE;
    }

    return getTransaction().resolveToTableName(getCurrentSchema(), name,
                                               isInCaseInsensitiveMode());

  }

  /**
   * Returns the DataTableDef for the table with the given name.
   */
  public DataTableDef getDataTableDef(TableName name) {
    name = substituteReservedTableName(name);
    return getTransaction().getDataTableDef(name);
  }

  /**
   * Returns a DataTable that represents the table from the given schema,
   * name in the database.
   */
  public DataTable getTable(TableName name) {
    name = substituteReservedTableName(name);

    try {
      // Special handling of NEW and OLD table, we cache the DataTable in the
      // OldNewTableState object,
      if (name.equals(Database.OLD_TRIGGER_TABLE)) {
        if (current_old_new_state.OLD_data_table == null) {
          current_old_new_state.OLD_data_table =
                          new DataTable(this, getTransaction().getTable(name));
        }
        return current_old_new_state.OLD_data_table;
      }
      else if (name.equals(Database.NEW_TRIGGER_TABLE)) {
        if (current_old_new_state.NEW_data_table == null) {
          current_old_new_state.NEW_data_table =
                          new DataTable(this, getTransaction().getTable(name));
        }
        return current_old_new_state.NEW_data_table;
      }

      // Ask the transaction for the table
      MutableTableDataSource table = getTransaction().getTable(name);

      // Is this table in the tables_cache?
      DataTable dtable = (DataTable) tables_cache.get(table);
      // No, so wrap it around a Datatable and put it in the cache
      if (dtable == null) {
        dtable = new DataTable(this, table);
        tables_cache.put(table, dtable);
      }
      // Return the DataTable
      return dtable;

    }
    catch (DatabaseException e) {
      Debug().writeException(e);
      throw new Error("Database Exception: " + e.getMessage());
    }

  }

  /**
   * Returns a DataTable that represents the table with the given name in the
   * database from the current connection schema.
   */
  public DataTable getTable(String table_name) {
    return getTable(new TableName(current_schema, table_name));
  }

  /**
   * Create a new table within the context of the current connection
   * transaction.
   */
  public void createTable(DataTableDef table_def) {
    checkAllowCreate(table_def.getTableName());
    getTransaction().createTable(table_def);
  }

  /**
   * Create a new table with a starting initial sector size.  This should
   * only be used as very fine grain optimization for creating tables.  If
   * in the future the underlying table model is changed so that the given
   * 'sector_size' value is unapplicable, then the value will be ignored.
   */
  public void createTable(DataTableDef table_def,
                          int data_sector_size, int index_sector_size) {
    checkAllowCreate(table_def.getTableName());
    getTransaction().createTable(table_def,
                                 data_sector_size, index_sector_size);
  }

  /**
   * Creates a new view.  This takes the information in the ViewDef object and
   * adds it to the system view table.
   * <p>
   * Note that this is a transactional operation.  You need to commit for the
   * view to be visible to other transactions.
   */
  public void createView(SQLQuery query, ViewDef view) {
    checkAllowCreate(view.getDataTableDef().getTableName());

    try {
      view_manager.defineView(view, query, getUser());
    }
    catch (DatabaseException e) {
      Debug().writeException(e);
      throw new RuntimeException("Database Exception: " + e.getMessage());
    }

  }

  /**
   * Drops the view with the given name and returns true if the drop succeeded.
   * Returns false if the view was not found.
   * <p>
   * Note that this is a transactional operation.  You need to commit for the
   * change to be visible to other transactions.
   */
  public boolean dropView(TableName view_name) {

    try {
      return view_manager.deleteView(view_name);
    }
    catch (DatabaseException e) {
      Debug().writeException(e);
      throw new RuntimeException("Database Exception: " + e.getMessage());
    }

  }
  
  /**
   * Updates a given table within the context of the current connection
   * transaction.
   */
  public void updateTable(DataTableDef table_def) {
    checkAllowCreate(table_def.getTableName());
    getTransaction().alterTable(table_def.getTableName(), table_def);
  }

  /**
   * Updates a given table within the context of the current connection
   * transaction.  This should only be used as very fine grain optimization
   * for creating tables.If in the future the underlying table model is
   * changed so that the given 'sector_size' value is unapplicable, then the
   * value will be ignored.
   */
  public void updateTable(DataTableDef table_def,
                          int data_sector_size, int index_sector_size) {
    checkAllowCreate(table_def.getTableName());
    getTransaction().alterTable(table_def.getTableName(), table_def,
                                data_sector_size, index_sector_size);
  }

  /**
   * Given a DataTableDef, if the table exists then it is updated otherwise
   * if it doesn't exist then it is created.
   * <p>
   * This should only be used as very fine grain optimization for creating/
   * altering tables.  If in the future the underlying table model is changed
   * so that the given 'sector_size' value is unapplicable, then the value
   * will be ignored.
   */
  public void alterCreateTable(DataTableDef table_def,
                               int data_sector_size, int index_sector_size) {
    if (!tableExists(table_def.getTableName())) {
      createTable(table_def, data_sector_size, index_sector_size);
    }
    else {
      updateTable(table_def, data_sector_size, index_sector_size);
    }
  }

  /**
   * Given a DataTableDef, if the table exists then it is updated otherwise
   * if it doesn't exist then it is created.
   */
  public void alterCreateTable(DataTableDef table_def) {
    if (!tableExists(table_def.getTableName())) {
      createTable(table_def);
    }
    else {
      updateTable(table_def);
    }
  }

  /**
   * Notifies this transaction that a database object with the given name has
   * successfully been created.
   */
  void databaseObjectCreated(TableName table_name) {
    getTransaction().databaseObjectCreated(table_name);
  }

  /**
   * Notifies this transaction that a database object with the given name has
   * successfully been dropped.
   */
  void databaseObjectDropped(TableName table_name) {
    getTransaction().databaseObjectDropped(table_name);
  }

  /**
   * Checks all the rows in the table for immediate constraint violations
   * and when the transaction is next committed check for all deferred
   * constraint violations.  This method is used when the constraints on a
   * table changes and we need to determine if any constraint violations
   * occurred.  To the constraint checking system, this is like adding all
   * the rows to the given table.
   */
  public void checkAllConstraints(TableName table_name) {
    // Assert
    checkExclusive();
    getTransaction().checkAllConstraints(table_name);
  }

  /**
   * Drops a table from within the context of the current connection
   * transaction.
   */
  public void dropTable(String table_name) {
    dropTable(new TableName(current_schema, table_name));
  }

  /**
   * Drops a table from within the context of the current connection
   * transaction.
   */
  public void dropTable(TableName table_name) {
    getTransaction().dropTable(table_name);
  }

  /**
   * Compacts the table with the given name.  Throws an exception if the
   * table doesn't exist.
   */
  public void compactTable(String table_name) {
    compactTable(new TableName(current_schema, table_name));
  }

  /**
   * Compacts the table with the given name.  Throws an exception if the
   * table doesn't exist.
   */
  public void compactTable(TableName table_name) {
    getTransaction().compactTable(table_name);
  }

  /**
   * Adds the given table name to the list of tables that are selected from
   * within the transaction in this connection.
   */
  public void addSelectedFromTable(String table_name) {
    addSelectedFromTable(new TableName(current_schema, table_name));
  }

  /**
   * Adds the given table name to the list of tables that are selected from
   * within the transaction in this connection.
   */
  public void addSelectedFromTable(TableName name) {
    getTransaction().addSelectedFromTable(name);
  }

  /**
   * Requests of the sequence generator the next value from the sequence.
   * <p>
   * NOTE: This does NOT check that the user owning this connection has the
   * correct privs to perform this operation.
   */
  public long nextSequenceValue(String name) {
    // Resolve and ambiguity test
    TableName seq_name = resolveToTableName(name);
    return getTransaction().nextSequenceValue(seq_name);
  }

  /**
   * Returns the current sequence value for the given sequence generator that
   * was last returned by a call to 'nextSequenceValue'.  If a value was not
   * last returned by a call to 'nextSequenceValue' then a statement exception
   * is generated.
   * <p>
   * NOTE: This does NOT check that the user owning this connection has the
   * correct privs to perform this operation.
   */
  public long lastSequenceValue(String name) {
    // Resolve and ambiguity test
    TableName seq_name = resolveToTableName(name);
    return getTransaction().lastSequenceValue(seq_name);
  }

  /**
   * Sets the sequence value for the given sequence generator.  If the generator
   * does not exist or it is not possible to set the value for the generator
   * then an exception is generated.
   * <p>
   * NOTE: This does NOT check that the user owning this connection has the
   * correct privs to perform this operation.
   */
  public void setSequenceValue(String name, long value) {
    // Resolve and ambiguity test
    TableName seq_name = resolveToTableName(name);
    getTransaction().setSequenceValue(seq_name, value);
  }
  
  /**
   * Returns the next unique identifier for the given table from the schema.
   */
  public long nextUniqueID(TableName name) {
    return getTransaction().nextUniqueID(name);
  }

  /**
   * Returns the next unique identifier for the given table in the connection
   * schema.
   */
  public long nextUniqueID(String table_name) {
    TableName tname = TableName.resolve(current_schema, table_name);
    return nextUniqueID(tname);
  }

  /**
   * If the given table name is a reserved name, then we must substitute it
   * with its correct form.  For example, 'APP.NEW' becomes 'SYS_INFO.NEW',
   * etc.
   */
  static TableName substituteReservedTableName(TableName table_name) {
    // We do not allow tables to be created with a reserved name
    String name = table_name.getName();
    if (name.equalsIgnoreCase("OLD")) {
      return Database.OLD_TRIGGER_TABLE;
    }
    if (name.equalsIgnoreCase("NEW")) {
      return Database.NEW_TRIGGER_TABLE;
    }
    return table_name;
  }
  
  /**
   * Generates an exception if the name of the table is reserved and the
   * creation of the table should be prevented.  For example, the table
   * names 'OLD' and 'NEW' are reserved.
   */
  static void checkAllowCreate(TableName table_name) {
    // We do not allow tables to be created with a reserved name
    String name = table_name.getName();
    if (name.equalsIgnoreCase("OLD") ||
        name.equalsIgnoreCase("NEW")) {
      throw new StatementException("Table name '" + table_name +
                                   "' is reserved.");
    }
  }
  
  /**
   * Creates a new sequence generator with the given TableName and
   * initializes it with the given details.  This does NOT check if the
   * given name clashes with an existing database object.
   */
  public void createSequenceGenerator(
               TableName name, long start_value, long increment_by,
               long min_value, long max_value, long cache, boolean cycle) {

    // Check the name of the database object isn't reserved (OLD/NEW)
    checkAllowCreate(name);

    getTransaction().createSequenceGenerator(name,
               start_value, increment_by, min_value, max_value, cache, cycle);
  }

  /**
   * Drops an existing sequence generator with the given name.
   */
  public void dropSequenceGenerator(TableName name) {
    getTransaction().dropSequenceGenerator(name);
  }
  
  /**
   * Adds a type of trigger for the given trigger source (usually the
   * name of the table).
   * <p>
   * Adds a type of trigger to the given Table.  When the event is fired, the
   * UserCallBack method is notified of the event.
   */
  public void createTrigger(String trigger_name,
                            String trigger_source, int type) {
    database.getTriggerManager().addTriggerListener(
                              this, trigger_name, type, trigger_source, this);
  }

  /**
   * Removes a type of trigger for the given trigger source (usually the
   * name of the table).
   */
  public void deleteTrigger(String trigger_name) {
    database.getTriggerManager().removeTriggerListener(this, trigger_name);
  }

  /**
   * Informs the underlying transaction that a high level transaction event
   * has occurred and should be dispatched to any listeners occordingly.
   */
  public void notifyTriggerEvent(TriggerEvent evt) {
    trigger_event_list.add(evt);
  }

  /**
   * Allocates a new large object in the Blob store of this conglomerate of the
   * given type and size.  The blob data must be written through the
   * Ref after the large object is created.  Once the data has been written
   * the 'complete' method in Ref is called.
   * <p>
   * Once a large object is created and written to, it may be allocated in one
   * or more tables in the conglomerate.
   */
  public Ref createNewLargeObject(byte type, long object_size) {
    // Enable compression for string types (but not binary types).
    if (type == 3 || type == 4) {
      type = (byte) (type | 0x010);
    }
    return conglomerate.createNewLargeObject(type, object_size);
  }

  /**
   * Tells the conglomerate to flush the blob store.  This should be called
   * after one or more blobs have been created and the data for the blob(s) are
   * set.  It is an important step to perform AFTER blobs have been written.
   * <p>
   * If this is not called and the database closes (or crashes) before a flush
   * occurs then the blob may not be recoverable.
   */
  public void flushBlobStore() {
    conglomerate.flushBlobStore();
  }

  /**
   * Returns a TableQueryDef object that describes the characteristics of a
   * table including the name (TableName), the columns (DataTableDef) and the
   * query plan to produce the table (QueryPlanNode).  This object can be used
   * to resolve information about a particular table, and to evaluate the
   * query plan to produce the table itself.
   * <p>
   * This produces TableQueryDef objects for all table objects in the database
   * including data tables and views.
   * <p>
   * The 'aliased_as' parameter is used to overwrite the default name of the
   * table object.
   */
  public TableQueryDef getTableQueryDef(final TableName table_name,
                                        final TableName aliased_as) {

    // Produce the data table def for this database object.
    DataTableDef dtf = getDataTableDef(table_name);
    // If the table is aliased, set a new DataTableDef with the given name
    if (aliased_as != null) {
      dtf = new DataTableDef(dtf);
      dtf.setTableName(aliased_as);
      dtf.setImmutable();
    }
    final DataTableDef data_table_def = dtf;
//    final String aliased_name =
//                     aliased_as == null ? null : aliased_as.getName();

    return new TableQueryDef() {
      public DataTableDef getDataTableDef() {
        return data_table_def;
      }
      public QueryPlanNode getQueryPlanNode() {
        return createObjectFetchQueryPlan(table_name, aliased_as);
      }
    };
                                          
  }

  /**
   * Creates a QueryPlanNode to fetch the given table object from this
   * connection.
   */
  public QueryPlanNode createObjectFetchQueryPlan(TableName table_name,
                                                  TableName aliased_name) {
    String table_type = getTableType(table_name);
    if (table_type.equals("VIEW")) {
      return new QueryPlan.FetchViewNode(table_name, aliased_name);
    }
    else {
      return new QueryPlan.FetchTableNode(table_name, aliased_name);
    }
  }
  
  // ---------- Schema management and constraint methods ----------
  // Methods that handle getting/setting schema information such as;
  // * Creating/dropping/querying schema
  // * Creating/dropping/querying constraint information including;
  //     check constraints, unique constraints, primary key constraints,
  //     foreign key constraints, etc.

  /**
   * Changes the default schema to the given schema.
   */
  public void setDefaultSchema(String schema_name) {
    boolean ignore_case = isInCaseInsensitiveMode();
    SchemaDef schema = resolveSchemaCase(schema_name, ignore_case);
    if (schema == null) {
      throw new Error("Schema '" + schema_name + "' does not exist.");
    }
    else {
      // Set the default schema for this connection
      setCurrentSchema(schema.getName());
    }
  }

  // NOTE: These methods are copied because they simply call through to the
  //   Transaction implementation of the method with the same signature.

  private void checkExclusive() {
    if (!getLockingMechanism().isInExclusiveMode()) {
      throw new Error("Assertion failed: Expected to be in exclusive mode.");
    }
  }

  /**
   * Same as the Transaction.createSchema method.
   */
  public void createSchema(String name, String type) {
    // Assert
    checkExclusive();
    getTransaction().createSchema(name, type);
  }

  /**
   * Same as the Transaction.dropSchema method.
   */
  public void dropSchema(String name) {
    // Assert
    checkExclusive();
    getTransaction().dropSchema(name);
  }

  /**
   * Same as the Transaction.schemaExists method.
   */
  public boolean schemaExists(String name) {
    return getTransaction().schemaExists(name);
  }

  /**
   * Same as the Transaction.resolveSchemaCase method.
   */
  public SchemaDef resolveSchemaCase(String name, boolean ignore_case) {
    return getTransaction().resolveSchemaCase(name, ignore_case);
  }

  /**
   * Convenience - returns the SchemaDef object given the name of the schema.
   * If identifiers are case insensitive, we resolve the case of the schema
   * name also.
   */
  public SchemaDef resolveSchemaName(String name) {
    boolean ignore_case = isInCaseInsensitiveMode();
    return resolveSchemaCase(name, ignore_case);
  }
  
  /**
   * Same as the Transaction.getSchemaList method.
   */
  public SchemaDef[] getSchemaList() {
    return getTransaction().getSchemaList();
  }
  
  /**
   * Same as the Transaction.setPersistentVar method.
   */
  public void setPersistentVar(String variable, String value) {
    // Assert
    checkExclusive();
    getTransaction().setPersistentVar(variable, value);
  }

  /**
   * Same as the Transaction.getPersistentVar method.
   */
  public String getPersistentVar(String variable) {
    return getTransaction().getPersistantVar(variable);
  }

  /**
   * Same as the Transaction.addUniqueConstraint method.
   */
  public void addUniqueConstraint(TableName table_name, String[] cols,
                                  short deferred, String constraint_name) {
    // Assert
    checkExclusive();
    getTransaction().addUniqueConstraint(table_name, cols,
                                         deferred, constraint_name);
  }

  /**
   * Same as the Transaction.addForeignKeyConstraint method.
   */
  public void addForeignKeyConstraint(TableName table, String[] cols,
                                      TableName ref_table, String[] ref_cols,
                                      String delete_rule, String update_rule,
                                      short deferred, String constraint_name) {
    // Assert
    checkExclusive();
    getTransaction().addForeignKeyConstraint(table, cols, ref_table, ref_cols,
                                             delete_rule, update_rule,
                                             deferred, constraint_name);
  }

  /**
   * Same as the Transaction.addPrimaryKeyConstraint method.
   */
  public void addPrimaryKeyConstraint(TableName table_name, String[] cols,
                                      short deferred, String constraint_name) {
    // Assert
    checkExclusive();
    getTransaction().addPrimaryKeyConstraint(table_name, cols,
                                             deferred, constraint_name);
  }

  /**
   * Same as the Transaction.addCheckConstraint method.
   */
  public void addCheckConstraint(TableName table_name,
               Expression expression, short deferred, String constraint_name) {
    // Assert
    checkExclusive();
    getTransaction().addCheckConstraint(table_name, expression,
                                        deferred, constraint_name);
  }

  /**
   * Same as the Transaction.dropAllConstraintsForTable method.
   */
  public void dropAllConstraintsForTable(TableName table_name) {
    // Assert
    checkExclusive();
    getTransaction().dropAllConstraintsForTable(table_name);
  }

  /**
   * Same as the Transaction.dropNamedConstraint method.
   */
  public int dropNamedConstraint(TableName table_name,
                                 String constraint_name) {
    // Assert
    checkExclusive();
    return getTransaction().dropNamedConstraint(table_name, constraint_name);
  }

  /**
   * Same as the Transaction.dropPrimaryKeyConstraintForTable method.
   */
  public boolean dropPrimaryKeyConstraintForTable(
                              TableName table_name, String constraint_name) {
    // Assert
    checkExclusive();
    return getTransaction().dropPrimaryKeyConstraintForTable(table_name,
                                                             constraint_name);
  }

  /**
   * Same as the Transaction.queryTablesRelationallyLinkedTo method.
   */
  public TableName[] queryTablesRelationallyLinkedTo(TableName table) {
    return Transaction.queryTablesRelationallyLinkedTo(getTransaction(), table);
  }

  /**
   * Same as the Transaction.queryTableUniqueGroups method.
   */
  public Transaction.ColumnGroup[] queryTableUniqueGroups(
                                               TableName table_name) {
    return Transaction.queryTableUniqueGroups(getTransaction(), table_name);
  }

  /**
   * Same as the Transaction.queryTablePrimaryKeyGroup method.
   */
  public Transaction.ColumnGroup queryTablePrimaryKeyGroup(
                                               TableName table_name) {
    return Transaction.queryTablePrimaryKeyGroup(getTransaction(), table_name);
  }

  /**
   * Same as the Transaction.queryTableCheckExpression method.
   */
  public Transaction.CheckExpression[] queryTableCheckExpressions(
                                               TableName table_name) {
    return Transaction.queryTableCheckExpressions(getTransaction(), table_name);
  }

  /**
   * Same as the Transaction.queryTableForeignKeyReferences method.
   */
  public Transaction.ColumnGroupReference[] queryTableForeignKeyReferences(
                                                      TableName table_name) {
    return Transaction.queryTableForeignKeyReferences(getTransaction(),
                                                      table_name);
  }

  /**
   * Same as the Transaction.queryTableImportedForeignKeyReferences method.
   */
  public Transaction.ColumnGroupReference[]
               queryTableImportedForeignKeyReferences(TableName table_name) {
    return Transaction.queryTableImportedForeignKeyReferences(getTransaction(),
                                                              table_name);
  }


  // ---------- Triggered OLD/NEW table handling ----------
  // These methods are used by the ConnectionTriggerManager object to
  // temporarily create OLD and NEW tables in this connection from inside a
  // triggered action.  In some cases (before the operation) the OLD table
  // is mutable.

  /**
   * Returns the current state of the old/new tables.
   */
  OldNewTableState getOldNewTableState() {
    return current_old_new_state;
  }

  /**
   * Sets the current state of the old/new tables.  When nesting OLD/NEW
   * tables for nested stored procedures, the current state should be first
   * recorded and reverted back when the nested procedure finishes.
   */
  void setOldNewTableState(OldNewTableState state) {
    current_old_new_state = state;
  }
  
  // ---------- Trigger methods ----------

  /**
   * Notifies this connection that an insert/delete or update operation has
   * occurred on some table of this DatabaseConnection.  This should notify
   * the trigger connection manager of this event so that it may perform any
   * action that may have been set up to occur on this event.
   */
  void fireTableEvent(TableModificationEvent evt) {
    connection_trigger_manager.performTriggerAction(evt);
  }

  // ---------- Implemented from TriggerListener ----------

  /**
   * Notifies when a trigger has fired for this user.  If there are no open
   * transactions on this connection then we do a straight call back trigger
   * notify.  If there is a transaction open then trigger events are added
   * to the 'trigger_event_buffer' which fires when the connection transaction
   * is committed or rolled back.
   */
  public void fireTrigger(DatabaseConnection database, String trigger_name,
                          TriggerEvent evt) {

    if (this != database) {
      throw new Error("User object mismatch.");
    }

    try {
      // Did we pass in a call back interface?
      if (call_back != null) {
        synchronized (trigger_event_buffer) {
          // If there is no active transaction then fire trigger immediately.
          if (transaction == null) {
            call_back.triggerNotify(trigger_name, evt.getType(),
                                    evt.getSource(), evt.getCount());
          }
          // Otherwise add to buffer
          else {
            trigger_event_buffer.add(trigger_name);
            trigger_event_buffer.add(evt);
          }
        }
      }
    }
    catch (Throwable e) {
      Debug().write(Lvl.ERROR, this, "TRIGGER Exception: " + e.getMessage());
    }
  }

  /**
   * Fires any triggers that are pending in the trigger buffer.
   */
  private void firePendingTriggerEvents() {
    int sz;
    synchronized (trigger_event_buffer) {
      sz = trigger_event_buffer.size();
    }
    if (sz > 0) {
      // Post an event that fires the triggers for each listener.
      Runnable runner = new Runnable() {
        public void run() {
          synchronized (trigger_event_buffer) {
            // Fire all pending trigger events in buffer
            for (int i = 0; i < trigger_event_buffer.size(); i += 2) {
              String trigger_name = (String) trigger_event_buffer.get(i);
              TriggerEvent evt =
                          (TriggerEvent) trigger_event_buffer.get(i + 1);
              call_back.triggerNotify(trigger_name, evt.getType(),
                                      evt.getSource(), evt.getCount());
            }
            // Clear the buffer
            trigger_event_buffer.clear();
          }
        }
      };

      // Post the event to go off approx 3ms from now.
      database.postEvent(3, database.createEvent(runner));
    }

  }

  /**
   * Private method that disposes the current transaction.
   */
  private void disposeTransaction() {
    // Set the transaction to null
    transaction = null;
    // Fire any pending trigger events in the trigger buffer.
    firePendingTriggerEvents();
    // Clear the trigger events in this object
    trigger_event_list.clear();
    
    // Notify any table backed caches that this transaction has finished.
    int sz = table_backed_cache_list.size();
      for (int i = 0; i < sz; ++i) {
      TableBackedCache cache =
                           (TableBackedCache) table_backed_cache_list.get(i);
      cache.transactionFinished();
    }
  }

  /**
   * Tries to commit the current transaction.  If the transaction can not be
   * committed because there were concurrent changes that interfered with
   * each other then a TransactionError is thrown and the transaction is
   * rolled back.
   * <p>
   * NOTE: It's guarenteed that the transaction will be closed even if a
   *   transaction exception occurs.
   * <p>
   * Synchronization is implied on this method, because the locking mechanism
   *   should be exclusive when this is called.
   */
  public void commit() throws TransactionException {
    // Are we currently allowed to commit/rollback?
    if (close_transaction_disabled) {
      throw new RuntimeException("Commit is not allowed.");
    }
    
    if (user != null) {
      user.refreshLastCommandTime();
    }

    // NOTE, always connection exclusive op.
    getLockingMechanism().reset();
    tables_cache.clear();

    if (transaction != null) {
      try {

        // Close and commit the transaction
        transaction.closeAndCommit();

        // Fire all SQL action level triggers that were generated on actions.
        database.getTriggerManager().flushTriggerEvents(trigger_event_list);
        
        
      }
      finally {
        // Dispose the current transaction
        disposeTransaction();
      }
    }
  }

  /**
   * Rolls back the current transaction operating within this connection.
   * <p>
   * NOTE: It's guarenteed that the transaction will be closed even if an
   *   exception occurs.
   * <p>
   * Synchronization is implied on this method, because the locking mechanism
   *   should be exclusive when this is called.
   */
  public void rollback() {
    // Are we currently allowed to commit/rollback?
    if (close_transaction_disabled) {
      throw new RuntimeException("Rollback is not allowed.");
    }

    if (user != null) {
      user.refreshLastCommandTime();
    }

    // NOTE, always connection exclusive op.
    tables_cache.clear();

    if (transaction != null) {
      getLockingMechanism().reset();
      try {
        transaction.closeAndRollback();
      }
      finally {
        // Dispose the current transaction
        disposeTransaction();
        // Dispose the jdbc connection
        if (jdbc_connection != null) {
          try {
            InternalJDBCHelper.disposeJDBCConnection(jdbc_connection);
          }
          catch (Throwable e) {
            Debug().write(Lvl.ERROR, this,
                          "Error disposing internal JDBC connection.");
            Debug().writeException(Lvl.ERROR, e);
            // We don't wrap this exception
          }
          jdbc_connection = null;
        }
      }
    }
  }

  /**
   * Closes this database connection.
   */
  public void close() {
    try {
      rollback();
    }
    catch (Throwable e) {
      e.printStackTrace(System.err);
    }
    finally {
      if (table_backed_cache_list != null) {
        try {
          int sz = table_backed_cache_list.size();
          for (int i = 0; i < sz; ++i) {
            TableBackedCache cache =
                           (TableBackedCache) table_backed_cache_list.get(i);
            cache.detatchFrom(conglomerate);
          }
          table_backed_cache_list = null;
        }
        catch (Throwable e) {
          e.printStackTrace(System.err);
        }
      }
      // Remove any trigger listeners set for this connection,
      database.getTriggerManager().clearAllDatabaseConnectionTriggers(this);
    }
  }


  public void finalize() throws Throwable {
    super.finalize();
    close();
  }

  // ---------- Inner classes ----------

  /**
   * An implementation of ProcedureConnection generated from this object.
   */
  private class DCProcedureConnection implements ProcedureConnection {
    
    /**
     * The User of this connection before this procedure was started.
     */
    private User previous_user;
    
    /**
     * The 'close_transaction_disabled' flag when this connection was created.
     */
    private boolean transaction_disabled_flag;

    /**
     * The JDBCConnection created by this object.
     */
    private java.sql.Connection jdbc_connection;


    public java.sql.Connection getJDBCConnection() {
      if (jdbc_connection == null) {
        jdbc_connection = InternalJDBCHelper.createJDBCConnection(getUser(),
                                                     DatabaseConnection.this);
      }
      return jdbc_connection;
    }

    public Database getDatabase() {
      return DatabaseConnection.this.getDatabase();
    }
    

    void dispose() {
      previous_user = null;
      if (jdbc_connection != null) {
        try {
          InternalJDBCHelper.disposeJDBCConnection(jdbc_connection);
        }
        catch (Throwable e) {
          Debug().write(Lvl.ERROR, this,
                        "Error disposing internal JDBC connection.");
          Debug().writeException(Lvl.ERROR, e);
          // We don't wrap this exception
        }
      }
    }

  }

  /**
   * An internal table info object that handles OLD and NEW tables for
   * triggered actions.
   */
  private class OldAndNewInternalTableInfo implements InternalTableInfo {

    private boolean hasOLDTable() {
      return current_old_new_state.OLD_row_index != -1;
    }
    
    private boolean hasNEWTable() {
      return current_old_new_state.NEW_row_data != null;
    }

    public int getTableCount() {
      int count = 0;
      if (hasOLDTable()) {
        ++count;
      }
      if (hasNEWTable()) {
        ++count;
      }
      return count;
    }

    public int findTableName(TableName name) {
      if (hasOLDTable() && name.equals(Database.OLD_TRIGGER_TABLE)) {
        return 0;
      }
      if (hasNEWTable() && name.equals(Database.NEW_TRIGGER_TABLE)) {
        if (hasOLDTable()) {
          return 1;
        }
        else {
          return 0;
        }
      }
      return -1;
    }

    public TableName getTableName(int i) {
      if (hasOLDTable()) {
        if (i == 0) {
          return Database.OLD_TRIGGER_TABLE;
        }
      }
      return Database.NEW_TRIGGER_TABLE;
    }

    public boolean containsTableName(TableName name) {
      return findTableName(name) != -1;
    }

    public String getTableType(int i) {
      return "SYSTEM TABLE";
    }

    public DataTableDef getDataTableDef(int i) {
      DataTableDef table_def = DatabaseConnection.this.getDataTableDef(
                                        current_old_new_state.trigger_source);
      DataTableDef new_table_def = new DataTableDef(table_def);
      new_table_def.setTableName(getTableName(i));
      return new_table_def;
    }

    public MutableTableDataSource createInternalTable(int index) {
      DataTableDef t_def = getDataTableDef(index);

      TriggeredOldNewDataSource table =
                            new TriggeredOldNewDataSource(getSystem(), t_def);

      if (hasOLDTable()) {
        if (index == 0) {

          // Copy data from the table to the new table
          DataTable dtable = DatabaseConnection.this.getTable(
                                        current_old_new_state.trigger_source);
          RowData old_row_data = new RowData(table);
          int row_index = current_old_new_state.OLD_row_index;
          for (int i = 0; i < t_def.columnCount(); ++i) {
            old_row_data.setColumnDataFromTObject(i,
                                        dtable.getCellContents(i, row_index));
          }
          // All OLD tables are immutable
          table.setImmutable(true);
          table.setRowData(old_row_data);
          
          return table;
        }
      }

      table.setImmutable(!current_old_new_state.mutable_NEW);
      table.setRowData(current_old_new_state.NEW_row_data);

      return table;
    }

  }
  
  /**
   * A MutableTableDataSource implementation that is used for trigger actions
   * to represent the data in the OLD and NEW tables.
   */
  private static class TriggeredOldNewDataSource extends GTDataSource {

    private DataTableDef table_def;

    private RowData content;

    private boolean immutable;
    
    /**
     * Constructor.
     */
    public TriggeredOldNewDataSource(TransactionSystem system,
                                     DataTableDef table_def) {
      super(system);
      this.table_def = table_def;
    }

    void setImmutable(boolean im) {
      this.immutable = im;
    }

    void setRowData(RowData row_data) {
      this.content = row_data;
    }
    
    public DataTableDef getDataTableDef() {
      return table_def;
    }
    
    public int getRowCount() {
      return 1;
    }
    
    public TObject getCellContents(final int column, final int row) {
      if (row < 0 || row > 0) {
        throw new RuntimeException("Row index out of bounds.");
      }
      return content.getCellData(column);
    }
    
    public int addRow(RowData row_data) {
      throw new RuntimeException("Inserting into table '" +
                    getDataTableDef().getTableName() + "' is not permitted.");
    }

    public void removeRow(int row_index) {
      throw new RuntimeException("Deleting from table '" +
                    getDataTableDef().getTableName() + "' is not permitted.");
    }
 
    public int updateRow(int row_index, RowData row_data) {
      if (immutable) {
        throw new RuntimeException("Updating table '" +
                    getDataTableDef().getTableName() + "' is not permitted.");
      }
      if (row_index < 0 || row_index > 0) {
        throw new RuntimeException("Row index out of bounds.");
      }

      int sz = getDataTableDef().columnCount();
      for (int i = 0; i < sz; ++i) {
        content.setColumnDataFromTObject(i, row_data.getCellData(i));
      }

      return 0;
    }

    public MasterTableJournal getJournal() {
      // Shouldn't be used...
      throw new RuntimeException("Invalid method used.");
    }

    public void flushIndexChanges() {
      // Shouldn't be used...
      throw new RuntimeException("Invalid method used.");
    }
  
    public void constraintIntegrityCheck() {
      // Should always pass (not integrity check needed for OLD/NEW table.
    }

  }

  /**
   * A list of DataTableDef system table definitions for tables internal to
   * the database connection.
   */
  private final static DataTableDef[] INTERNAL_DEF_LIST;

  static {
    INTERNAL_DEF_LIST = new DataTableDef[5];
    INTERNAL_DEF_LIST[0] = GTStatisticsDataSource.DEF_DATA_TABLE_DEF;
    INTERNAL_DEF_LIST[1] = GTConnectionInfoDataSource.DEF_DATA_TABLE_DEF;
    INTERNAL_DEF_LIST[2] = GTCurrentConnectionsDataSource.DEF_DATA_TABLE_DEF;
    INTERNAL_DEF_LIST[3] = GTSQLTypeInfoDataSource.DEF_DATA_TABLE_DEF;
    INTERNAL_DEF_LIST[4] = GTPrivMapDataSource.DEF_DATA_TABLE_DEF;
  }

  /**
   * An internal table info object that handles tables internal to a
   * DatabaseConnection object.
   */
  private class ConnectionInternalTableInfo extends AbstractInternalTableInfo {

    /**
     * Constructor.
     */
    public ConnectionInternalTableInfo() {
      super("SYSTEM TABLE", INTERNAL_DEF_LIST);
    }

    // ---------- Implemented ----------

    public MutableTableDataSource createInternalTable(int index) {
      if (index == 0) {
        return new GTStatisticsDataSource(DatabaseConnection.this).init();
      }
      else if (index == 1) {
        return new GTConnectionInfoDataSource(DatabaseConnection.this).init();
      }
      else if (index == 2) {
        return new GTCurrentConnectionsDataSource(
                                              DatabaseConnection.this).init();
      }
      else if (index == 3) {
        return new GTSQLTypeInfoDataSource(DatabaseConnection.this).init();
      }
      else if (index == 4) {
        return new GTPrivMapDataSource(DatabaseConnection.this);
      }
      else {
        throw new RuntimeException();
      }
    }

  }

  /**
   * Call back interface for events that occur within the connection instance.
   */
  public static interface CallBack {

    /**
     * Notifies the callee that a trigger event was fired that this user
     * is listening for.
     */
    void triggerNotify(String trigger_name, int trigger_event,
                       String trigger_source, int fire_count);

  }

  /**
   * An object that stores state about the trigger table OLD and NEW when
   * the connection is set up to execute a stored procedure.
   */
  static class OldNewTableState {
    
    /**
     * The name of the table that is the trigger source.
     */
    TableName trigger_source;

    /**
     * The row index of the OLD data that is being updated or deleted in the
     * trigger source table.
     */
    int OLD_row_index = -1;

    /**
     * The RowData of the new data that is being inserted/updated in the trigger
     * source table.
     */
    RowData NEW_row_data;

    /**
     * If true then the 'new_data' information is mutable which would be true for
     * a BEFORE trigger.  For example, we would want to change the data in the
     * row that caused the trigger to fire.
     */
    boolean mutable_NEW;

    /**
     * The DataTable object that represents the OLD table, if set.
     */
    DataTable OLD_data_table;

    /**
     * The DataTable object that represents the NEW table, if set.
     */
    DataTable NEW_data_table;

    /**
     * Constructor.
     */
    OldNewTableState(TableName table_source,
                     int old_d, RowData new_d, boolean is_mutable) {
      this.trigger_source = table_source;
      this.OLD_row_index = old_d;
      this.NEW_row_data = new_d;
      this.mutable_NEW = is_mutable;
    }

    /**
     * Default constructor.
     */
    OldNewTableState() {
    }
    
  }
  
}
