/*
 * This file is subject to the licence found in LICENCE.TXT in the root directory of the project.
 * Copyright Jayasoft 2005 - All rights reserved
 * 
 * version 1.3.1
 */
package fr.jayasoft.ivy.repository;

import java.io.File;
import java.io.IOException;
import java.util.List;

/**
 * Represents a collection of resources available to Ivy. Ivy uses one or more
 * repositories as both a source of resources for Ivy enabled build systems and
 * as a distribution center for resources generated by Ivy enabled build systems.
 * </p>
 * <p>A repository supports the following fundamental operations
 * <ul>
 *   <li>retrieving a resource from the repository.</li>
 *   <li>transfering a resource to the repository.</li>
 *   <li>retrieving a listing of resources.</li>
 * </ul>
 * </p>
 * <h4>Resource Retrieval</h4>
 * </p>
 * <p>{@link #get} retrieves a resource specified by a provided identifier creating a new file .</p>
 * </p>
 * <h4>resource Publication</h4>
 * </p>
 * <p>{@link #put} transfers a file to the repository.
 * </p>
 * </p>
 * <h4>resource Listing</h4>
 * </p>
 * <p>{@link #list} returns a listing of file like objects
 *    belonging to a specified parent directory.</p>
 * </p>
 */
public interface Repository {
	
	/**
	 * Return the resource associated with a specified identifier.
	 * 
	 * @param source A string identifying the resource.
	 * @return The resource associated with the resource identifier.
	 * @throws IOException On failure to get resource.
	 */
    Resource getResource(String source)  throws IOException;
    
    /**
     * Fetch a resource from the repository.
     * 
     * @param source A string identifying the resource to be fetched.
     * @param destination Where to place the fetched resource.
     * @throws IOException On retrieval failure.
     */
    void get(String source, File destination) throws IOException;
    
    
    /**
     * Transfer a resource to the repository
     * 
     * @param source The local file to be transferred.
     * @param destination Where to transfer the resource.
     * @param overwrite Whether the transfer should overwrite an existing resource.
     * @throws IOException On publication failure.
     */
    void put(File source, String destination, boolean overwrite) throws IOException;
    
    /**
     * Return a listing of resources
     * 
     * @param parent The parent directory from which to generate the listing.
     * @return A listing of the parent directory's file content.
     * @throws IOException On listing failure.
     */
    List list(String parent) throws IOException;
    
    /**
     * Add a listener to the repository.
     * 
     * @param listener The listener to attach to the repository.
     */
    void addTransferListener(TransferListener listener);
    
    /**
     * Remove a listener on the repository
     * 
     * @param listener The listener to remove
     */
    void removeTransferListener(TransferListener listener);
    
    /**
     * Determine if a given listener is attached to the repository.
     * 
     * @param listener The listener being quireied
     * @return <code>true</code> if the provided listener is attached to the repository,
     *         <code>false</code> if not.
     */
    boolean hasTransferListener(TransferListener listener);
    
    /**
     * Get the repository's file separator string.
     * 
     * @return The repository's file separator delimiter
     */
    String getFileSeparator();
    
    /**
     * Normalize a string.
     * 
     * @param source The string to normalize.
     * @return The normalized string.
     */
    String standardize(String source);
    
    /**
     * Return the name of the repository
     * 
     */
    String getName();
}
